/*
 Erica Sadun, http://ericasadun.com
 iPhone Developer's Cookbook, 5.0 Edition
 BSD License, Use at your own risk
 */

#import "DownloadHelper.h"

#define SAFE_PERFORM_WITH_ARG(THE_OBJECT, THE_SELECTOR, THE_ARG) (([THE_OBJECT respondsToSelector:THE_SELECTOR]) ? [THE_OBJECT performSelector:THE_SELECTOR withObject:THE_ARG] : nil)

@implementation DownloadHelper
@synthesize delegate, urlString, targetPath;
@synthesize isDownloading, bytesRead, expectedLength;

- (void) start
{
	isDownloading = NO;
    if (!urlString)
    {
        NSLog(@"Nie podano wymaganego ciągu tekstowego adresu URL.");
        return;
    }
	
	NSURL *url = [NSURL URLWithString:urlString];
	if (!url)
	{
		NSString *reason = [NSString stringWithFormat:@"Nie można utworzyć adresu URL na podstawie ciągu tekstowego %@", urlString];
		SAFE_PERFORM_WITH_ARG(delegate, @selector(dataDownloadFailed:), reason);
		return;
	}
	
	NSMutableURLRequest *theRequest = [NSMutableURLRequest requestWithURL:url];
	if (!theRequest)
	{
		NSString *reason = [NSString stringWithFormat:@"Nie można utworzyć żądania URL na podstawie ciągu tekstowego %@", urlString];
		SAFE_PERFORM_WITH_ARG(delegate, @selector(dataDownloadFailed:), reason);
		return;
	}
	
	urlconnection = [[NSURLConnection alloc] initWithRequest:theRequest delegate:self];
	if (!urlconnection)
	{
		NSString *reason = [NSString stringWithFormat:@"Połączenie URL dla ciągu tekstowego %@ zakończyło się niepowodzeniem.", urlString];
		SAFE_PERFORM_WITH_ARG(delegate, @selector(dataDownloadFailed:), reason);
		return;
	}
	
	outputStream = [[NSOutputStream alloc] initToFileAtPath:targetPath append:YES];
	if (!outputStream)
	{
		NSString *reason = [NSString stringWithFormat:@"Nie można utworzyć strumienia wynikowego w lokalizacji %@", targetPath];
		SAFE_PERFORM_WITH_ARG(delegate, @selector(dataDownloadFailed:), reason);
		return;
	}
	[outputStream open];
	
	isDownloading = YES;
	bytesRead = 0;
	
	NSLog(@"Rozpoczęto pobieranie danych.");
	[urlconnection scheduleInRunLoop:[NSRunLoop currentRunLoop] forMode:NSRunLoopCommonModes];
}

- (void) cleanup
{
	isDownloading = NO;
    if (urlconnection)
    {
        [urlconnection cancel];
        urlconnection = nil;
    }
    
    if (outputStream)
    {
        [outputStream close];
        outputStream = nil;
    }
    
    self.urlString = nil;
	self.targetPath = nil;
}

- (void) dealloc
{
    [self cleanup];
}

- (void) cancel
{
    [self cleanup];
}

- (void)connection:(NSURLConnection *)connection didReceiveResponse:(NSURLResponse *)aResponse
{
	// Sprawdzenie, czy połączenie jest prawidłowe.
	expectedLength = [aResponse expectedContentLength];
	if (expectedLength == NSURLResponseUnknownLength)
	{
		NSString *reason = [NSString stringWithFormat:@"Nieprawidłowy adres URL [%@]", urlString];
		SAFE_PERFORM_WITH_ARG(delegate, @selector(dataDownloadFailed:), reason);
		[connection cancel];
		[self cleanup];
		return;
	}
}

- (void)connection:(NSURLConnection *)connection didReceiveData:(NSData *)theData
{
	bytesRead += theData.length;
	NSUInteger bytesLeft = theData.length;
	NSUInteger bytesWritten = 0;
	do {
		bytesWritten = [outputStream write:theData.bytes maxLength:bytesLeft];
		if (-1 == bytesWritten) break;
		bytesLeft -= bytesWritten;
	} while (bytesLeft > 0);
	if (bytesLeft) {
		NSLog(@"Błąd strumienia: %@", [outputStream streamError]);
	}

	SAFE_PERFORM_WITH_ARG(delegate, @selector(downloadReceivedData), nil);
}

- (void)connectionDidFinishLoading:(NSURLConnection *)connection
{
	// finished downloading the data, cleaning up
	[outputStream close];
	[urlconnection unscheduleFromRunLoop:[NSRunLoop currentRunLoop] forMode:NSRunLoopCommonModes];
	[self cleanup];
	
	SAFE_PERFORM_WITH_ARG(delegate, @selector(downloadFinished), nil);
}

- (void)connection:(NSURLConnection *)connection didFailWithError:(NSError *)error
{
	isDownloading = NO;
	NSLog(@"Błąd: nie udało się nawiązać połączenia, %@", [error localizedFailureReason]);
	SAFE_PERFORM_WITH_ARG(delegate, @selector(dataDownloadFailed:), @"Nieprawidłowe połączenie.");
	[self cleanup];
}

+ (id) download:(NSString *) aURLString withTargetPath: (NSString *) aPath withDelegate: (id <DownloadHelperDelegate>) aDelegate
{
    if (!aURLString)
    {
        NSLog(@"Błąd: brak ciągu tekstowego adresu URL.");
        return nil;
    }
    
    if (!aPath)
    {
        NSLog(@"Błąd: brak ścieżki dostępu do miejsca zapisu danych.");
        return nil;
    }
    
    DownloadHelper *helper = [[self alloc] init];
    helper.urlString = aURLString;
    helper.targetPath = aPath;
    helper.delegate = aDelegate;
    [helper start];
    
    return helper;
}
@end
